/* Configuration pour l'archivage des diffrents lments du fichier source   */
/******************************************************************************/
// !CONFIG!=/L/* /R/* /W"* Nom : "
// Dfinition du systme       !CONFIG!=/V1!EMULATEUR CPC!
// Dfinition du sous systme  !CONFIG!=/V2!WIN-CPC!
// Dfinition du sous ensemble !CONFIG!=/V3!Chips!
// Dfinition du module        !CONFIG!=/V4!UPD 765!
/******************************************************************************/

/********************************************************* !NAME! **************
* !./FLE!
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\Fichiers
********************************************************** !0! *****************
* ------------------------------------------------------------------------------
*          SYSTEME         |      SOUS SYSTEME       |      SOUS ENSEMBLE
* ------------------------------------------------------------------------------
*  EMULATEUR CPC           | WIN-CPC                 | Chips
* ------------------------------------------------------------------------------
*  Fichier     : UPD.C                 | Version : 0.1af
* ------------------------------------------------------------------------------
*  Date        : 05/11/2002            | Auteur  : L.DEPLANQUE
* ------------------------------------------------------------------------------
*  Description : Emulation du UPD 765
*
* ------------------------------------------------------------------------------
*  Historique  :
*           Date           |         Auteur          |       Description
* ------------------------------------------------------------------------------
*  05/11/2002              | L.DEPLANQUE             | creation
* ------------------------------------------------------------------------------
*  06/02/2003              | L.DEPLANQUE             | Version 0.1v : mmorise
*                          |                         | la longueur du fichier lu
*                          |                         | pour rcrire
*                          |                         | si ncessaire
* ------------------------------------------------------------------------------
*  18/02/2003              | L.DEPLANQUE             | Version 0.1w : prise en
*                          |                         | compte de l'information
*                          |                         | disc missing dans le
*                          |                         | registre ST0
* ------------------------------------------------------------------------------
*  04/03/2004              | L.DEPLANQUE             | Version 0.1ab :
*                          |                         | Correction d'un bugg
*                          |                         | dans la sauvegarde de la
*                          |                         | dernire disquette
*                          |                         | modifie.
* ------------------------------------------------------------------------------
*  07/03/2005              | L.DEPLANQUE             | Version 0.1af :
*                          |                         | Correction d'un bugg
*                          |                         | dans la gestion du lecteur
*                          |                         | dans les commandes
* ------------------------------------------------------------------------------
********************************************************** !END! **************/


#include  <string.h>
#include  <stdio.h>

#include  "types.h"
#ifdef USE_LOG
#include  "log.h"
#endif


//
// Constantes generales...
//
#define SECTSIZE   512
#define NBSECT     9


// Bits de Status
#define STATUS_CB       0x10
#define STATUS_NDM      0x20
#define STATUS_DIO      0x40
#define STATUS_RQM      0x80

// ST0
#define ST0_NR          0x08
#define ST0_SE          0x20
#define ST0_IC1         0x40
#define ST0_IC2         0x80

// ST1
#define ST1_ND          0x04
#define ST1_EN          0x80

// ST3
#define ST3_TS          0x08
#define ST3_T0          0x10
#define ST3_RY          0x20


/********************************************************* !NAME! **************
* Nom : ImgDsk
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : ImgDsk de stockage du fichier image disquette
*
********************************************************** !0! ****************/
static UBYTE ImgDsk[ 1024 * 1024 ];


/********************************************************* !NAME! **************
* Nom : CPCEMUEnt
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Structures
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : En-tte d'un fichier DSK
*
********************************************************** !0! ****************/
typedef struct
    {
    char  debut[ 0x30 ]; // "MV - CPCEMU Disk-File\r\nDisk-Info\r\n"
    UBYTE NbTracks;
    UBYTE NbHeads;
    UBYTE DataSizeLow;
    UBYTE DataSizeHigh; // 0x1300 = 256 + ( 512 * nbsecteurs )
    UBYTE Unused[ 0xCC ];
    } CPCEMUEnt;


/********************************************************* !NAME! **************
* Nom : CPCEMUSect
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Structures
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Description d'un secteur
*
********************************************************** !0! ****************/
typedef struct
    {
    UBYTE C;                // track
    UBYTE H;                // head
    UBYTE R;                // sect
    UBYTE N;                // size
    UBYTE ST1;              // Valeur ST1
    UBYTE ST2;              // Valeur ST2
    UBYTE SectSizeLow;
    UBYTE SectSizeHigh;     // Taille du secteur en octets
    } CPCEMUSect;


/********************************************************* !NAME! **************
* Nom : CPCEMUTrack
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Structures
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Description d'une piste
*
********************************************************** !0! ****************/
typedef struct
    {
    char        ID[ 0x10 ];   // "Track-Info\r\n"
    UBYTE       Track;
    UBYTE       Head;
    SHORT       Unused;
    UBYTE       SectSize; // 2
    UBYTE       NbSect;   // 9
    UBYTE       Gap3;     // 0x4E
    UBYTE       OctRemp;  // 0xE5
    CPCEMUSect  Sect[ 29 ];
    } CPCEMUTrack;


typedef int ( * pfctUPD )( int );


static int Rien( int );


/********************************************************* !NAME! **************
* Nom : fct
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Pointeur de fonction du UPD
*
********************************************************** !0! ****************/
static pfctUPD fct = Rien;


/********************************************************* !NAME! **************
* Nom : etat
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Compteur phase fonction UPD
*
********************************************************** !0! ****************/
static int etat;


/********************************************************* !NAME! **************
* Nom : CurrTrackDatasDSK
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Image d'une piste
*
********************************************************** !0! ****************/
static CPCEMUTrack CurrTrackDatasDSK;


/********************************************************* !NAME! **************
* Nom : Infos
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : En-tte du fichier image
*
********************************************************** !0! ****************/
static CPCEMUEnt Infos;


/********************************************************* !NAME! **************
* Nom : FlagWrite
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Indique si criture sur disquette effectue
*
********************************************************** !0! ****************/
static int FlagWrite;


/********************************************************* !NAME! **************
* Nom : FlagWrite
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Contient la longueur du fichier disquette lu
*
********************************************************** !0! ****************/
static int LongFic;


/********************************************************* !NAME! **************
* Nom : Image
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Indique si fichier image charg
*
********************************************************** !0! ****************/
static int Image;


/********************************************************* !NAME! **************
* Nom : PosData
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Position des donnes dans l'image
*
********************************************************** !0! ****************/
static int PosData;


/********************************************************* !NAME! **************
* Nom : NomFic
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Nom du fichier image ouvert
*
********************************************************** !0! ****************/
static char NomFic[ 256 ];


/********************************************************* !NAME! **************
* Nom : Status
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Status UPD
*
********************************************************** !0! ****************/
static int Status;


/********************************************************* !NAME! **************
* Nom : ST0
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Status interne 0
*
********************************************************** !0! ****************/
static int ST0;


/********************************************************* !NAME! **************
* Nom : ST1
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Status interne 1
*
********************************************************** !0! ****************/
static int ST1;


/********************************************************* !NAME! **************
* Nom : ST2
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Status interne 2
*
********************************************************** !0! ****************/
static int ST2;


/********************************************************* !NAME! **************
* Nom : ST3
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Status interne 3
*
********************************************************** !0! ****************/
static int ST3;


/********************************************************* !NAME! **************
* Nom : C
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Cylindre (n piste)
*
********************************************************** !0! ****************/
static int C;


/********************************************************* !NAME! **************
* Nom : H
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Head     (n tte)
*
********************************************************** !0! ****************/
static int H;


/********************************************************* !NAME! **************
* Nom : R
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Record   (n secteur)
*
********************************************************** !0! ****************/
static int R;


/********************************************************* !NAME! **************
* Nom : N
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Number   (nbre d'octet poids forts)
*
********************************************************** !0! ****************/
static int N;


/********************************************************* !NAME! **************
* Nom : Drive
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Drive    ( 0=A, 1=B)
*
********************************************************** !0! ****************/
static int Drive;


/********************************************************* !NAME! **************
* Nom : EOT
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Secteur final;
*
********************************************************** !0! ****************/
static int EOT;


/********************************************************* !NAME! **************
* Nom : Busy
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : UPD occup
*
********************************************************** !0! ****************/
static int Busy;


/********************************************************* !NAME! **************
* Nom : Inter
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Gnration d'une interruption UPD
*
********************************************************** !0! ****************/
static int Inter;


/********************************************************* !NAME! **************
* Nom : Moteur
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Variables Globales
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Etat moteur UPD
*
********************************************************** !0! ****************/
static int Moteur;


/********************************************************* !NAME! **************
* Nom : RechercheSecteur
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Recherche un secteur sur la piste courrante
*
* Rsultat    : L'index du secteur trouv
*
* Variables globales modifies : ST0, ST1
*
********************************************************** !0! ****************/
static int RechercheSecteur( int newSect, int * pos )
{
    int i;

    * pos = 0;
    for ( i = 0; i < CurrTrackDatasDSK.NbSect; i++ )
        if ( CurrTrackDatasDSK.Sect[ i ].R == newSect )
            return( ( UBYTE )i );
        else
            * pos += ( CurrTrackDatasDSK.Sect[ i ].SectSizeHigh << 8 );

    ST0 |= ST0_IC1;
    ST1 |= ST1_ND;
#ifdef USE_LOG
    sprintf( MsgLog, "secteur (C:%02X,H:%02X,R:%02X) non trouv", C, H, R );
    Log( MsgLog, LOG_DEBUG );
#endif
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : SetST0
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Calcul des registres ST0, ST1, ST2
*
* Rsultat    : /
*
* Variables globales modifies : ST0, ST1, ST2
*
********************************************************** !0! ****************/
static void SetST0( void )
{
    ST0 = 0; // drive A
    if ( ! Moteur || Drive || ! Image )
        ST0 |= ST0_IC1 | ST0_NR;

    ST1 = 0;
    ST2 = 0;
}


/********************************************************* !NAME! **************
* Nom : Rien
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction non reconnue du UPD
*
* Rsultat    : /
*
* Variables globales modifies : etat, ST0
*
********************************************************** !0! ****************/
static int Rien( int val )
{
    Status &= ~STATUS_CB & ~STATUS_DIO;
    etat = 0;
    ST0 = ST0_IC2;
#ifdef USE_LOG
    Log( "Appel fonction FDC Rien", LOG_DEBUG );
#endif
    return( val );
}


/********************************************************* !NAME! **************
* Nom : ReadST0
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD ReadST0
*
* Rsultat    : ST0, C
*
* Variables globales modifies : etat, ST0, ST1, Inter, Busy
*
********************************************************** !0! ****************/
static int ReadST0( int val )
{
    if ( ! Inter )
        ST0 = ST0_IC2;
    else
        {
        Inter = 0;
        if ( Busy )
            {
            ST0 = ST0_SE;
            Busy = 0;
            }
        else
            ST0 |= ST0_IC1 | ST0_IC2;
        }
    if ( Moteur && Image && ! Drive )
        ST0 &= ~ST0_NR;
    else
        {
        ST0 |= ST0_NR;
        if ( ! Image )
            ST0 |= ( ST0_IC1 | ST0_IC2 );
        }

    if ( etat++ == 1 )
        {
        Status |= STATUS_DIO;
        return( ST0 );
        }

    etat = val = 0;
    Status &= ~STATUS_CB & ~STATUS_DIO;
    ST0 &= ~ST0_IC1 & ~ST0_IC2;
    ST1 &= ~ST1_ND;
    return( C );
}


/********************************************************* !NAME! **************
* Nom : ReadST3
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD ReadST3
*
* Rsultat    : 0, ST3
*
* Variables globales modifies : etat, Status, ST3
*
********************************************************** !0! ****************/
static int ReadST3( int val )
{
    if ( etat++ == 1 )
        {
        Drive = val;
        Status |= STATUS_DIO;
        return( 0 );
        }
    etat = 0;
    Status &= ~STATUS_CB & ~STATUS_DIO;
    if ( Moteur && ! Drive )
        ST3 |= ST3_RY;
    else
        ST3 &= ~ST3_RY;

    return( ST3 );
}


/********************************************************* !NAME! **************
* Nom : Specify
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD Specify, non mule
*
* Rsultat    : 0
*
* Variables globales modifies : etat, Status
*
********************************************************** !0! ****************/
static int Specify( int val )
{
    if ( etat++ == 1 )
        return( 0 );

    etat = val = 0;
    Status &= ~STATUS_CB & ~STATUS_DIO;
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : ReadID
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD ReadID
*
* Rsultat    : ST0, ST1, ST2, C, H, R, N
*
* Variables globales modifies : Drive, Status, Inter, etat
*
********************************************************** !0! ****************/
static int ReadID( int val )
{
    switch( etat++ )
        {
        case 1 :
            Drive = val;
            Status |= STATUS_DIO;
            Inter = 1;
            break;

        case 2 :
            return( 0 /*ST0*/ );

        case 3 :
            return( ST1 );

        case 4 :
            return( ST2 );

        case 5 :
            return( C );

        case 6 :
            return( H );

        case 7 :
            return( R );

        case 8 :
            etat = 0;
            Status &= ~STATUS_CB & ~STATUS_DIO;
            return( N );
        }
    return( 0 );
}



/********************************************************* !NAME! **************
* Nom : FormatTrack
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD FormatTrack, non mule
*
* Rsultat    : 0
*
* Variables globales modifies : etat, Status
*
********************************************************** !0! ****************/
static int FormatTrack( int val )
{
#ifdef USE_LOG
    Log( "Appel fonction FDC format", LOG_DEBUG );
#endif
    etat = val = 0;
    Status &= ~STATUS_CB & ~STATUS_DIO;
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : Scan
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD Scan, non mule
*
* Rsultat    : 0
*
* Variables globales modifies : etat
*
********************************************************** !0! ****************/
static int Scan( int val )
{
#ifdef USE_LOG
    Log( "Appel fonction FDC scan", LOG_DEBUG );
#endif
    etat = val = 0;
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : ChangeCurrTrack
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Changement de la piste courrante
*
* Rsultat    : /
*
* Variables globales modifies : C, H, R, N, ST3
*
********************************************************** !0! ****************/
static void ChangeCurrTrack( int newTrack )
{
    ULONG Pos = 0;
    int t, s;

    if ( ! Infos.DataSizeHigh )
        {
        memcpy( &CurrTrackDatasDSK, ImgDsk, sizeof( CurrTrackDatasDSK ) );
        for ( t = 0; t < newTrack; t++ )
            {
            for ( s = 0; s < CurrTrackDatasDSK.NbSect; s++ )
                Pos += ( CurrTrackDatasDSK.Sect[ s ].SectSizeHigh << 8 );

            Pos += sizeof( CurrTrackDatasDSK );
            memcpy( &CurrTrackDatasDSK, &ImgDsk[ Pos ], sizeof( CurrTrackDatasDSK ) );
            }
        }
    else
        Pos += ( Infos.DataSizeHigh << 8 ) * newTrack;

    memcpy( &CurrTrackDatasDSK, &ImgDsk[ Pos ], sizeof( CurrTrackDatasDSK ) );
    if ( ! strncmp( CurrTrackDatasDSK.ID, "Track-Info", 10 ) )
        {
        PosData = Pos + sizeof( CurrTrackDatasDSK );
        C = CurrTrackDatasDSK.Sect[ 0 ].C;
        H = CurrTrackDatasDSK.Sect[ 0 ].H;
        R = CurrTrackDatasDSK.Sect[ 0 ].R;
        N = CurrTrackDatasDSK.Sect[ 0 ].N;
        }
#ifdef USE_LOG
    else
        {
        sprintf( MsgLog, "Erreur dplacement piste %d", newTrack );
        Log( MsgLog, LOG_WARNING );
        }
#endif

    if ( ! newTrack )
        ST3 |= ST3_T0;
    else
        ST3 &= ~ST3_T0;
}


/********************************************************* !NAME! **************
* Nom : MoveTrack
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD MoveTrack
*
* Rsultat    : 0
*
* Variables globales modifies : etat, Drive, Status, Busy, Inter
*
********************************************************** !0! ****************/
static int MoveTrack( int val )
{
    switch( etat++ )
        {
        case 1 :
            Drive = val;
            SetST0();
            Status |= STATUS_NDM;
            break;

        case 2 :
            ChangeCurrTrack( C = val );
            etat = 0;
            Status &= ~STATUS_CB & ~STATUS_DIO & ~STATUS_NDM;
            Busy = 1;
            Inter = 1;
            break;
        }
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : MoveTrack0
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD MoveTrack0
*
* Rsultat    : 0
*
* Variables globales modifies : etat, C, Status, Busy, Inter
*
********************************************************** !0! ****************/
static int MoveTrack0( int val )
{
    Drive = val;
    ChangeCurrTrack( C = 0 );
    etat = 0;
    Status &= ~STATUS_CB & ~STATUS_DIO & ~STATUS_NDM;
    SetST0();
    Busy = 1;
    Inter = 1;
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : ReadData
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD ReadData
*
* Rsultat    : Datas, ST0, ST1, ST2, C, H, R, N
*
* Variables globales modifies : C, H, R, N, EOT, etat, Status
*
********************************************************** !0! ****************/
static int ReadData( int val )
{
    static int sect = 0, cntdata = 0, fin = 0, newPos, TailleSect;

    switch( etat++ )
        {
        case 1 :
            Drive = val;
            SetST0();
            break;

        case 2 :
            C = val;
            break;

        case 3 :
            H = val;
            break;

        case 4 :
            R = val;
            break;

        case 5 :
            N = val;
            break;

        case 6 :
            EOT = val;
            fin = ( ( EOT - R + 1 ) * CurrTrackDatasDSK.SectSize ) << 8;
            break;

        case 7 :
            sect = RechercheSecteur( R, &newPos );
            TailleSect = 128 << CurrTrackDatasDSK.Sect[ sect ].N;
            if ( ! newPos )
                cntdata = ( sect * CurrTrackDatasDSK.SectSize ) << 8;
            else
                cntdata = newPos;

            break;

        case 8 :
            Status |= STATUS_DIO | STATUS_NDM;
            break;

        case 9 :
            if ( ! ( ST0 & ST0_IC1 ) )
                {
                if ( --fin )
                    {
                    if ( --TailleSect )
                        etat--;
                    else
                        {
                        R++;
                        etat = 7;
                        }
                    }
                else
                    Status &= ~STATUS_NDM;
                
                return( ImgDsk[ PosData + cntdata++ ] );
                }
            Status &= ~STATUS_NDM;
            return( 0 );

        case 10 :
            return( ST0 );

        case 11 :
            return( ST1 | ST1_EN );       // ### ici PB suivant logiciels... ###

        case 12 :
            return( ST2 );

        case 13 :
            return( C );

        case 14 :
            return( H );

        case 15 :
            return( R );

        case 16 :
            etat = 0;
            Status &= ~STATUS_CB & ~STATUS_DIO;
            return( N );
        }
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : WriteData
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Fonction UPD WriteData
*
* Rsultat    : ST0, ST1, ST2, C, H, R, N
*
* Variables globales modifies : C, H, R, N, EOT, etat, Status
*
********************************************************** !0! ****************/
static int WriteData( int val )
{
    static int sect = 0, cntdata = 0, fin = 0, newPos, TailleSect;

    switch( etat++ )
        {
        case 1 :
            Drive = val;
            SetST0();
            break;

        case 2 :
            C = val;
            break;

        case 3 :
            H = val;
            break;

        case 4 :
            R = val;
            break;

        case 5 :
            N = val;
            break;

        case 6 :
            EOT = val;
            fin = ( ( EOT - R + 1 ) * CurrTrackDatasDSK.SectSize ) << 8;
            break;

        case 7 :
            sect = RechercheSecteur( R, &newPos );
            TailleSect = 128 << CurrTrackDatasDSK.Sect[ sect ].N;
            if ( ! newPos )
                cntdata = ( sect * CurrTrackDatasDSK.SectSize ) << 8;
            else
                cntdata = newPos;

            break;

        case 8 :
            Status |= STATUS_DIO | STATUS_NDM;
            break;

        case 9 :
            if ( ! ( ST0 & ST0_IC1 ) )
                {
                ImgDsk[ PosData + cntdata++ ] = ( UBYTE )val;
                if ( --fin )
                    {
                    if ( --TailleSect )
                        etat--;
                    else
                        {
                        R++;
                        etat = 7;
                        }
                    }
                else
                    Status &= ~STATUS_NDM;

                return( 0 );
                }
            Status &= ~STATUS_NDM;
            return( 0 );

        case 10 :
            if ( ! ( ST0 & ST0_IC1 ) )
                FlagWrite = 1;

            return( ST0 );

        case 11 :
            return( ST1 );

        case 12 :
            return( ST2 );

        case 13 :
            return( C );

        case 14 :
            return( H );

        case 15 :
            return( R );

        case 16 :
            etat = 0;
            Status &= ~STATUS_CB & ~STATUS_DIO;
            return( N );
        }
    return( 0 );
}


/********************************************************* !NAME! **************
* Nom : ReadUPD
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Lecture d'un registre/donne du UPD
*
* Rsultat    : Valeur registre/donne du UPD
*
* Variables globales modifies : /
*
********************************************************** !0! ****************/
int ReadUPD( int port )
{
    if ( port & 1 )
        return( fct( port ) );

    return( Status );
}


/********************************************************* !NAME! **************
* Nom : WriteUPD
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Ecriture d'un registre/donne du UPD
*
* Rsultat    : /
*
* Variables globales modifies : Status, etat, fct, Moteur
*
********************************************************** !0! ****************/
void WriteUPD( int port, int val )
{
    if ( port == 0xFB7F )
        {
        if ( etat )
            fct( val );
        else
            {
            Status |= STATUS_CB;
            etat = 1;
            switch( val & 0x1F )
                {
                case 0x03 :
                    // Specify
                    fct = Specify;
                    break;

                case 0x04 :
                    // Lecture ST3
                    fct = ReadST3;
                    break;

                case 0x05 :
                    // Ecriture donnes
                    fct = WriteData;
                    break;

                case 0x06 :
                    // Lecture donnes
                    fct = ReadData;
                    break;

                case 0x07 :
                    // Dplacement tte piste 0
//                    Status |= STATUS_NDM;
                    fct = MoveTrack0;
                    break;

                case 0x08 :
                    // Lecture ST0, track
                    Status |= STATUS_DIO;
                    fct = ReadST0;
                    break;

                case 0x09 :
                    // Ecriture donnes
                    fct = WriteData;
                    break;

                case 0x0A :
                    // Lecture champ ID
                    fct = ReadID;
                    break;

                case 0x0C :
                    // Lecture donnes
                    fct = ReadData;
                    break;

                case 0x0D :
                    // Formattage piste
                    fct = FormatTrack;
                    break;

                case 0x0F :
                    // Dplacement tte
                    fct = MoveTrack;
                    break;

                case 0x11 :
                    fct = Scan;
                    break;

                default :
#ifdef USE_LOG
                    sprintf( MsgLog
                           , "Accs fonction UPD inconnue : %02X"
                           , val
                           );
                    Log( MsgLog, LOG_DEBUG );
#endif
                    fct = Rien;
                }
            }
        }
    else
        if ( port == 0xFA7E )
            Moteur = val & 1;
}


/********************************************************* !NAME! **************
* Nom : ResetUPD
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Effectue un reset du UPD
*
* Rsultat    : /
*
* Variables globales modifies : Status, ST0, ST1, ST2, ST3, Busy, Inter, etat
*
********************************************************** !0! ****************/
void ResetUPD( void )
{
    Status = STATUS_RQM;
    ST0 = ST0_SE;
    ST1 = 0;
    ST2 = 0;
    ST3 = ST3_RY | ST3_TS;
    Busy = 0;
    Inter = 0;
    etat = 0;
}


/********************************************************* !NAME! **************
* Nom : EjectDiskUPD
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Termine l'accs  une disquette (ejection)
*
* Rsultat    : /
*
* Variables globales modifies : /
*
********************************************************** !0! ****************/
void EjectDiskUPD( void )
{
    FILE * handle;

    Image = 0;
    if ( FlagWrite )
        {
        // Ecriture de la disquette ???
/*
        if ( MessageBox( NULL
                       , "Sauvegarde fichier disquette ?"
                       , "Emu-CPC"
                       , MB_YESNO
                       ) == IDYES
            )
            {
            handle = fopen( NomFic, "wb" );
            if ( handle )
                {
                fwrite( &Infos, sizeof( Infos ), 1, handle );
                fwrite( ImgDsk, LongFic, 1, handle );
                fclose( handle );
                }
            }
        FlagWrite = 0;
*/
        }
}


/********************************************************* !NAME! **************
* Nom : SetDiskUPD
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Initialise l'accs  une disquette (insertion)
*
* Rsultat    : /
*
* Variables globales modifies : NomFic, Image, FlagWrite
*
********************************************************** !0! ****************/
void SetDiskUPD( char * n )
{
    FILE * handle;

    EjectDiskUPD();
    strcpy( NomFic, n );
    handle = fopen( NomFic, "rb" );
    if ( handle )
        {
        fread( &Infos, sizeof( Infos ), 1, handle );
        LongFic = fread( ImgDsk, 1, sizeof( ImgDsk ), handle );
        fclose( handle );
        Image = 1;
        }
    FlagWrite = 0;
    ChangeCurrTrack( 0 );
}


/********************************************************* !NAME! **************
* Nom : GetCurrTrack
********************************************************** !PATHS! *************
* !./V1!\!./V2!\!./V3!\!./V4!\Fonctions
********************************************************** !1! *****************
*
* Fichier     : !./FPTH\/FLE!, ligne : !./LN!
*
* Description : Retourne la piste courrante
*
* Rsultat    : C
*
* Variables globales modifies : /
*
********************************************************** !0! ****************/
int GetCurrTrack( void )
{
    return( C );
}
